package ZxingP5;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.oned.EAN13Writer;
import com.google.zxing.oned.EAN8Writer;

import processing.core.PApplet;
import processing.core.PImage;

//********************************************************
//EAN-Code Writer
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing EAN-Code Writer<br>
 * 本クラスは、指定された文字列をもとに、EAN-13またはEAN-8の画像イメージを作成
 * するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * 制約事項：<br>
 * ・数字（0～9)が利用可能<br>
 * ・桁数はEAN13なら13文字、EAN8なら8文字固定<br>
 * ・先頭3桁は国別コードで、仕様として許可されているものを指定する事<br>
 * ・チェックデジットは必須で、かならず付加される<br>
 * ・チェックデジット方式にはモジュラス10ウェイト3方式が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class EANWriter {
    private EAN13Writer writer13;
    private EAN8Writer writer8;
    private PImage rowImage;
    private BitMatrix bitMatrix;
    private PApplet app;

    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの書き込み準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter;  //EAN-Code Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public EANWriter(PApplet applet) {
        this.writer13 = new EAN13Writer();
        this.writer8 = new EAN8Writer();
        this.app = applet;
    }

    /**
     * バーコード拡大縮小<br>
     * 画像化したバーコードを拡大縮小します。<br>
     * 一度もEAN画像を生成していない場合、null が戻されます。<br><br>
     *
     * @param width 画像の横幅（0以上の値）
     * @param height 画像の高さ（0以上の値）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter; //EAN-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     *   //画像化
     *   if( eanWriter.encode8( "4501234", 200, 100 ) != null ){
     *     //画像拡大
     *     pImage = eanWriter.resize(300,150);
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage resize(int width, int height) {
    	PImage pImage = null;
    	if( rowImage != null ) {
    		pImage = ZxingP5Com.resizeImage(rowImage, width, height);
    	}
    	return pImage;
    }

    /**
     * バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 文字列に使用するキャラクターコードは、EAN13またはEAN8の仕様に準拠している必要があります<br><br>
     * widthに0を与えた場合や画像化に必要な幅より小さな値の場合、必要な幅は自動的に計算されます。<br><br>
     * appFont に true を指定すると、バーコードの下にコード化した文字列を付加します。デフォルトは「付加しない」です。<br><br>
     * チェックデジットは自動的に付加されます。またチェックデジット付きの文字列が与えられた場合で、かつチェックデジットが
     * 不正な場合は、正しい値に置き換えられます。<br><br>
     *
     * @param src 画像化したい文字列
     * @param format バーコード指定（BarcodeFormat.EAN_8、EAN_13）
     * @param width 画像の横幅（0以上の値）
     * @param height 画像の高さ（5以上の値）
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter; //EAN-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     *   //画像化
     *   pImage = eanWriter.encode( "450123456789",BarcodeFormat.EAN_13, 200, 100 );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode(String src, BarcodeFormat format, int width, int height, boolean ...appFont ) {
    	//パラメータチェツク
        if( !ZxingP5Com.checkEANParam(format, src, width, height)) {
        	ZxingP5Com.logout( String.format("パラメータ不正"));
        	return null;
        }

        //元々のチェックデジットを削除する
        src = ZxingP5Com.eanDeleteCheckDegit(src);

        //画像化
    	try {
    		//チェックデジットは Zxing に自動計算させて画像化する
        	if( format == BarcodeFormat.EAN_13) {
        		bitMatrix = this.writer13.encode(src, format, width, height);
        	}
        	else
        	{
        		bitMatrix = this.writer8.encode(src, format, width, height);
        	}
        } catch (WriterException e) {
        	ZxingP5Com.logout( String.format("バーコード生成で WriterException 例外発生" ));
        	return null;
        } catch( IllegalArgumentException e) {
        	ZxingP5Com.logout( String.format("バーコード生成で IllegalArgumentException 例外発生" ));
        	return null;
        }

    	//画像（BitMatrix）をPImageに変換する
    	rowImage  = ZxingP5Com.changeBitMatrix2PImage(bitMatrix);

    	PImage pImage = rowImage.copy();
    	if( appFont.length > 0 && appFont[0] == true ) {
    		pImage = ZxingP5Com.appendCodeImage(app, rowImage, src, format);
    	}

    	return(pImage);
    }

    /**
     * EAN13 バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 詳細は encode を参照して下さい。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter; //EAN-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     *   //画像化
     *   pImage = eanWriter.encode13( "450123456789", 200, 100 );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode13(String src,  int width, int height, boolean ...appFont){
    	return encode(src, BarcodeFormat.EAN_13, width, height, appFont);
    }

    /**
     * EAN8 バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 詳細は encode を参照して下さい。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter; //EAN-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     *   //画像化
     *   pImage = eanWriter.encode8( "4501234", 200, 100 );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode8(String src,  int width, int height, boolean ...appFont){
    	return encode(src, BarcodeFormat.EAN_8, width, height, appFont);
    }

    /**
     * バーコード画像取得<br>
     * 画像化されたイメージを取得します。<br>
     * 本メソッドで戻される画像イメージには、コード化された文字画像は含まれていません。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANWriter  eanWriter; //EAN-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanWriter = new EANWriter(this);
     *   //画像化
     *   if( eanWriter.encode8( "4501234", 200, 100 ) != null ){
     *     //画像取得
     *     pImage = eanWriter.getImage();
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
	public PImage getImage() {
		return rowImage;
    	//return ZxingP5Com.changeBitMatrix2PImage( bitMatrix );
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( EANWriter.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
