package ZxingP5;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.FormatException;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.oned.EAN13Reader;
import com.google.zxing.oned.EAN8Reader;

import processing.core.PApplet;
import processing.core.PImage;
import processing.core.PVector;

//********************************************************
//EAN-Code Reader
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing EAN-Code Reader<br>
 * 本クラスは、EAN-13またはEAN-8が描かれた画像から、画像を解析してコード内容
 * を取得するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * 制約事項：<br>
 * ・数字（0～9)が利用可能<br>
 * ・桁数はEAN13なら13文字、EAN8なら8文字固定<br>
 * ・先頭3桁は国別コードで、仕様として許可されているものを指定する事<br>
 * ・チェックデジットは必須で、かならず付加される<br>
 * ・チェックデジット方式にはモジュラス10ウェイト3方式が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class EANReader {

	private EAN13Reader reader13;
	private EAN8Reader  reader8;
	private Result result = null;
	private PApplet app;
    private int rotateDegree;
    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの読み取り準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader;  //EAN-Code Reader
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   eanReader = new EANReader(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
	public EANReader(PApplet applet) {
        this.reader13 = new EAN13Reader();
        this.reader8 = new EAN8Reader();
        this.app = applet;
        this.rotateDegree = 10;
    }

	/**
	 * 回転角度指定<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを指定します。<br>
	 * デフォルトは10度です。<br><br>
	 * @param degree 回転角度（1 から 180）
	 */
	public void setRotateDegree(int degree) {
		if( degree > 0 && degree < 181 ) {
			rotateDegree = degree;
		}
	}

	/**
	 * 回転角度取得<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを取得します。<br>
	 * デフォルトは10度です。<br><br>
	 * @return 回転角度（1 から 180）
	 */
	public int getRotateDegree() {
		return rotateDegree;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 本メソッドは内部処理用です。直接呼び出すことはできません。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br><br>
     *
     * @param format バーコード指定（BarcodeFormat.EAN_8、EAN_13）
     * @param bitmap EANが描画された画像
     * @return 解析結果：成功<br>null：失敗<br><br>
     */
	Result decodeRow(BarcodeFormat format, BinaryBitmap bitmap) {
		this.result = null;

		//解析モードを指定する(精度重視)
        Map<DecodeHintType,Object> hints = new HashMap<DecodeHintType, Object>();
        hints.put( DecodeHintType.TRY_HARDER, true );

        //バーコード解析
        try {
        	if( format == BarcodeFormat.EAN_8) {
        		this.result = reader8.decode(bitmap, hints);
        	}
        	else {
        		this.result = reader13.decode(bitmap, hints);
        	}
        } catch (NotFoundException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り NotFoundException 例外発生" ));
        } catch (FormatException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り FormatException 例外発生" ));
        }

        return this.result;
	}

    /**
     * EAN13 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * 読み取る対象画像が、あらかじめ EAN13 だとわかっている場合に利用します。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage EANが描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean13Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode13( pImage, true );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode13(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		PImage copyImage = pImage.copy();

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			//バーコード読み取り
			decodeRow( BarcodeFormat.EAN_13, bitmap);

	        if (this.result != null && this.result.getText() != null) {
	        	//成功
	            return this.result.getText();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate( app, pImage, degree );
	        }
	        else {
	        	break;
	        }
		}

        return null;
	}

    /**
     * EAN13 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * 読み取る対象画像が、あらかじめ EAN13 だとわかっている場合に利用します。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br><br>
     *
     * @param pImage EANが描画された画像
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean13Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode13( pImage );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode13(PImage pImage) {
        return this.decode13(pImage, true, false);
    }

    /**
     * EAN8 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * 読み取る対象画像が、あらかじめ EAN8 だとわかっている場合に利用します。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage EANが描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode8( pImage, true );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode8(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			PImage copyImage = pImage.copy();

			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			//バーコード読み取り
			this.result = decodeRow( BarcodeFormat.EAN_8, bitmap);

	        if (this.result != null && this.result.getText() != null) {
	        	//成功
	            return this.result.getText();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate(app, pImage, degree);
	        }
	        else {
	        	break;
	        }
		}

        return null;
	}

    /**
     * EAN8 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * 読み取る対象画像が、あらかじめ EAN8 だとわかっている場合に利用します。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br><br>
     *
     * @param pImage EANが描画された画像
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode8( pImage );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode8(PImage pImage) {
        return this.decode8(pImage, true, false);
    }

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * まず最初に画像をEAN13として読み取り、NGの場合はEAN8として読み取ります。<br><br>
     *
     * @param pImage EANが描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode( pImage, true );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		//EAN13として読み取ってみる
        String retText = decode13(pImage, globalHistogram, doRotate);
        if( retText == null) {
        	//NG なら EAN8として読み取ってみる
        	ZxingP5Com.logout("EAN13では無さそうです。EAN8として読み取ります。");

        	retText = decode8(pImage, globalHistogram, doRotate);
        }

        return retText;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * まず最初に画像をEAN13として読み取り、NGの場合はEAN8として読み取ります。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br><br>
     *
     * @param pImage EANが描画された画像
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   String text = eanReader.decode( pImage );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode(PImage pImage) {
        return this.decode(pImage, true, false);
    }

    /**
     * 頂点座標取得<br>
     * バーコード解析結果を使い、頂点座標を取得します。
     * EANの場合は、バーコードの左端と右端のセンター座標が得られます。<br>
     * 一度もEANを読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 座標配列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * PVector[]  points;    //頂点座標
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   eanReader.decode( pImage );
     *   //頂点座標取得
     *   points = eanReader.getPoints();
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     *   //シンボル座標に赤い○を描く
     *   fill(color(255,0,0));
     *   if( points != null ){
     *     for( PVector p : points ){
     *       ellipse( p.x, p.y, 10, 10 );
     *     }
     *   }
     * }
     * }
     * </pre>
     */
    public PVector[] getPoints() {
    	return ZxingP5Com.getPoints(this.result);
  	}

    /**
     * 解析結果取得<br>
     * バーコード解析結果を取得します。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br>
     * 一度もEANを読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 解析結果：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * EANReader eanReader; //EAN-Code Reader
     * PImage    pImage;    //画像
     * Result    result;    //解析結果
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("ean8Sample.png");
     *   //インスタンス作成
     *   eanReader = new EANReader();
     *   //画像解析
     *   eanReader.decode( pImage );
     *   //解析結果取得
     *   result = eanReader.getResult();
     *   if( result != null ){
     *     println( result.getText() );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public Result getResult() {
        return this.result;
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( EANReader.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
