package ZxingP5;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;

import javax.imageio.ImageIO;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.LuminanceSource;
import com.google.zxing.Result;
import com.google.zxing.ResultPoint;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.GlobalHistogramBinarizer;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;

import processing.core.PApplet;
import processing.core.PGraphics;
import processing.core.PImage;
import processing.core.PVector;

//********************************************************
//バーコード共通処理
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing Common Function<br>
 * 本クラスは、バーコードライブラリに関する共通処理です。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public final class ZxingP5Com {
	public final static String VERSION = "1.0.0";
	static boolean logMode = false;
	static FileWriter logWriter = null;
    static File canonPath = null;

    /**
     * 拡張子取得<br>
     * 指定されたファイルパスから拡張子を取得する。<br><br>
     *
     * @param fileName フルパス
     * @return 拡張子：成功<br>null：失敗/拡張子なし
     */
	static String getSuffix(String fileName) {
	    if (fileName == null)
	        return null;

	    //パスの後ろから . を探す
	    int point = fileName.lastIndexOf(".");
	    if (point != -1) {
	    	// . を見つけたら、そこまでが拡張子
	        return fileName.substring(point + 1).toLowerCase();
	    }

	    logout("拡張子が見つかりません");

	    return null;
	}

	/**
	 * ログ出力ON<br>
	 * エラー発生時、ログを採取します。<br>
	 * ONにされた以降に発生したエラーを、指定されたログファイルに記録します。<br>
	 * デフォルトは OFF です。<br><br>
	 *
	 * @param fileName 出力ログファイルの絶対パス<br><br>
	 * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   //ログ出力指定
     *   ZxingP5Com.setLogoutOn(dataPath("") + "\\logfile.log" );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
	 */
	public static void setLogoutOn(String fileName) {
		//拡張子がないなら、fileNameに .log を付加する
		if( getSuffix(fileName) == null ) {
			fileName = fileName + "." + ".log";
		}
		File outputFile = new File( fileName );

		//絶対パス化する
		try {
			canonPath = new File(outputFile.getCanonicalPath());
		} catch (IOException e) {
			logMode = false;
			return ;
		}

		//フォルダが存在するか検査する
        File folder = new File( canonPath.getParent());
        if( !folder.exists()) {
        	logMode = false;
        	return ;
        }

		logMode = true;
	}

	/**
	 * ログ出力OFF<br>
	 * ログの採取を停止します。<br><br>
	 *
	 * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   //ログ出力状態取得
     *   println( ZxingP5Com.getLogout() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
	 */
	public static void setLogoutOff() {
		logMode = false;
	}

	/**
	 * ログ出力モード取得<br>
	 * ログ出力を行うか否かを返却します。<br><br>
	 * @return True:出力する  False:出力しない<br><br>
	 *
	 * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   //ログ出力OFF
     *   ZxingP5Com.setLogoutOff();
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
	 */
	public static boolean getLogout() {
		return logMode;
	}

	/**
	 * ログ出力<br>
	 * @param src メッセージ
	 */
	static void logout(String src) {
		if(!logMode){
			return;
		}

        //クラス名とメソッド名を取得する
		StackTraceElement[] ste = Thread.currentThread().getStackTrace();
		String className = ste[2].getClassName();
		String mtoName = ste[2].getMethodName();

		//ログ出力時刻を取得する
		LocalDateTime ldt = LocalDateTime.now() ;
		DateTimeFormatter dtf1 =
			    DateTimeFormatter.ofPattern( "yyyy/MM/dd HH:mm:ss SSS" ) ;
		String  localStr = dtf1.format( ldt ) ;

		//ログを書き込む
		try{
			logWriter = new FileWriter(canonPath, true);
			PrintWriter pw = new PrintWriter(logWriter);
			pw.format("%s [%s:%s]:%s\r\n",localStr, className, mtoName, src);
			pw.close();
			logWriter.close();
		}catch(IOException e){
			logMode = false;
			return;
		}
	}

    /**
     * 画像保存処理<br>
     * 指定された画像をファイルに保存します。fileNameには絶対パスを指定して下さい。<br><br>
     *
     * @param pImage 保存画像
     * @param fileName 保存ファイルへの絶対パス
     * @return True：成功<br>False：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter();
     *   //画像化
     *   String code = "AB123XY";
     *   pImage = codeWriter.encode39( code, 150, 50, false ):
     *   if( pImage != null ){
     *     ZxingP5Com.SaveBarCodeImage(pImage, dataPath("") + "\\code39test.png" );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
	public static boolean saveBarCodeImage( PImage pImage, String fileName) {

		if( fileName == null || fileName.length() < 1 ) {
			logout( String.format("fileName引数が不正です"));
			return false;
		}

		if( pImage == null ) {
			logout( String.format("pImageがnullです"));
			return false;
		}

		//保存する
		boolean ret = saveBarCodeImage( (BufferedImage)pImage.getNative(), fileName);
		return ret;
	}

    /**
     * 画像保存処理<br>
     * 指定された画像をファイルに保存します。<br><br>
     *
     * @param img 保存画像
     * @param fileName フルパス
     * @return True：成功<br>False：失敗
     */
	static boolean saveBarCodeImage( BufferedImage img, String fileName) {
		String formatName = "png";
		File outputFile = null;

		//拡張子がないなら、fileNameに .png を付加する
		if( getSuffix(fileName) == null ) {
			fileName = fileName + "." + formatName;
		}
		outputFile = new File( fileName );

		//絶対パス化する
        File canonPath = null;
		try {
			canonPath = new File(outputFile.getCanonicalPath());
		} catch (IOException e) {
			logout( String.format("%s の絶対パス取得で IOException 例外発生" ));
			return false;
		}

		//フォルダが存在するか検査する
        File folder = new File( canonPath.getParent());
        if( !folder.exists()) {
        	logout( String.format("%s が存在しません", folder));
        	return false;
        }

        //画像を保存する
        try {
			ImageIO.write( img, formatName, canonPath);
		} catch (IOException e) {
			logout( String.format("%s ファイル出力で IOException 例外発生" ));
			return false;
		}
        return true;
	}

	/**
	 * 画像回転処理
	 * @param app PApplet
	 * @param pImage 画像
	 * @param degree 角度
	 * @return 画像
	 */
	static PImage codeImageRotate(PApplet app, PImage pImage, int degree) {

		if( degree == 0 || degree == 360 ) {
			return pImage;
		}

		//長辺を求めて、その1.42倍を新しい画像サイズにする
		int longLen = 0;
		if( pImage.width > pImage.height) {
			longLen = ((int)(pImage.width * 1.42));
		}
		else {
			longLen = ((int)(pImage.height * 1.42));
		}

		//画像を回転する
		PGraphics g = app.createGraphics(longLen, longLen);
		g.beginDraw();
		g.background(255);

		g.pushMatrix();
		g.translate( longLen/2, longLen/2 );
		g.rotate( PApplet.radians( degree ));
		g.imageMode(PApplet.CENTER);
		g.image( pImage, 0, 0 );
		g.imageMode(PApplet.CORNER);

		g.popMatrix();
		g.endDraw();

		PImage img = g.get();

		//DEBUG
		img.save(app.dataPath("") + "\\rotate" + String.valueOf(degree) + ".png" );

		return img;
	}

    /**
     * BinaryBitmap化処理<br>
     * PImageをBinaryBitmapに変換する。<br><br>
     *
     * @param pImage 対象画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @return BinaryBitmap：成功<br>null：失敗
     */
	static BinaryBitmap changePImage2Bitmap(PImage pImage, boolean globalHistogram) {
		//PImageからBufferedImageを得る
		BufferedImage buf = (BufferedImage) pImage.getNative();

		//BufferedImageをLuminanceSource化する
        LuminanceSource source = new BufferedImageLuminanceSource(buf);

        //LuminanceSourceからBinaryBitmapを作成する
        BinaryBitmap bitmap = null;

        if (globalHistogram) {
            bitmap = new BinaryBitmap(new GlobalHistogramBinarizer(source));
        } else {
            bitmap = new BinaryBitmap(new HybridBinarizer(source));
        }

        return bitmap;
	}

    /**
     * 拡大縮小処理<br>
     * PImageを指定サイズに変換する。<br><br>
     *
     * @param img 対象画像
     * @param width 新横幅
     * @param height 新高さ
     * @return PImage：成功<br>null：失敗
     */
	static PImage resizeImage(PImage src, int width, int height) {
		//幅と高さは1以上である事
		if( width < 1 || height < 1) {
			logout( String.format("画像の幅と高さは1以上で指定してください。 width = %d  height = %d", width, height));
    		return null;
		}

		//PImageからImageを得る
    	Image jImage = src.getImage();

    	Image scaledImg = null;
    	try {
    		//Imageを拡大縮小する
    		scaledImg = jImage.getScaledInstance(width, height, Image.SCALE_SMOOTH);
    	}
    	catch(IllegalArgumentException e) {
    		logout( String.format("拡大縮小処理で IllegalArgumentException 例外発生 width = %d  height = %d", width, height ));
    		return null;
    	}

    	return new PImage(scaledImg);
	}

    /**
     * PImage化処理<br>
     * BitMatrixをPImageに変換する。<br><br>
     *
     * @param img 対象画像
     * @return PImage：成功<br>null：失敗
     */
	static PImage changeBitMatrix2PImage( BitMatrix bitMatrix) {
		//変換する
		BufferedImage bImg = MatrixToImageWriter.toBufferedImage(bitMatrix);
		return new PImage( bImg );
	}

    /**
     * 座標取得処理<br>
     * 画像化結果より座標を取得する<br><br>
     *
     * @param result 変換結果
     * @return PVector[]：成功<br>null：失敗
     */
	static PVector[] getPoints(Result result) {
        if(result == null) {
        	logout( String.format("Resultがありません。"));
            return null;
        }

        ResultPoint[] resultPoints = result.getResultPoints();
        PVector[] points = new PVector[resultPoints.length];

        for (int i = 0; i < resultPoints.length; i++) {
            points[i] = new PVector(resultPoints[i].getX(), resultPoints[i].getY());
        }
        return points;
    }

    /**
     * QR-CODEパラメータ検査処理<br>
     * QR画像化できるか検査する<br><br>
     *
     * @param src 対象文字列
     * @param ecLevel 誤り訂正率
     * @param version バージョン(1 - 40)
     * @return True：成功<br>False：失敗
     */
	static boolean checkQRParam(String src, ErrorCorrectionLevel  ecLevel, int version) {
    	//バージョン情報チェツク
        if(1 > version || version > 40) {
        	logout( String.format("versionは1から40の範囲で指定して下さい。version = %d", version));
            return false;
        }

        //漢字をコード化する前提で、誤り訂正率と文字数から最適なバージョン番号を求め
        //それが指定バージョンより大きいか調べる
        int getVer = getQrVersion(src, ecLevel);

        if( getVer == 0 || getVer > version ) {
        	//指定バージョンが低すぎるか、文字が長すぎる
        	logout(String.format("変換対象文字列と誤り訂正率に見合う適切なQRバージョンが得られませんでした。文字列長 = %d  誤り訂正率 = %s", src.length(), ecLevel));
        	return false;
        }

        return true;
	}

	/**
	 * QRコード最適バージョン取得処理<br>
	 * 漢字をコード化する前提で、誤り訂正率と文字数から最適なバージョン番号を求めます。<br>br>
	 *
	 * @param src 対象文字列
	 * @param ecLevel 誤り訂正率
	 * @return 1-40：最適なバージョン番号  0：不正
	 */
	static int getQrVersion(String src, ErrorCorrectionLevel  ecLevel) {

		if( src == null || src.length() < 1) {
			logout( String.format("変換対象文字列が不正です。"));
			return 0;
		}

		//誤り訂正率毎に許されるMAX漢字文字数とVersionのテーブル
		//KEY = Version
		//DATA = L/M/Q/H 毎に許される漢字文字数
        HashMap <Integer, int[]> map = new HashMap <Integer, int[]>();
        map.put(1, new int[] {10,8,7,4});
        map.put(2, new int[] {20,16,12,8});
        map.put(3, new int[] {32,26,20,15});
        map.put(4, new int[] {148,38,28,21});
        map.put(5, new int[] {65,52,37,27});
        map.put(6, new int[] {82,65,45,36});
        map.put(7, new int[] {95,75,53,39});
        map.put(8, new int[] {118,93,66,52});
        map.put(9, new int[] {141,111,80,60});
        map.put(10, new int[] {167,131,93,74});
        map.put(11, new int[] {198,155,109,85});
        map.put(12, new int[] {226,177,125,96});
        map.put(13, new int[] {262,204,149,109});
        map.put(14, new int[] {282,223,159,120});
        map.put(15, new int[] {320,254,180,136});
        map.put(16, new int[] {361,277,198,154});
        map.put(17, new int[] {397,310,224,173});
        map.put(18, new int[] {442,345,243,191});
        map.put(19, new int[] {488,384,272,208});
        map.put(20, new int[] {528,410,297,235});
        map.put(21, new int[] {572,438,314,248});
        map.put(22, new int[] {618,480,348,270});
        map.put(23, new int[] {672,528,376,284});
        map.put(24, new int[] {721,561,407,315});
        map.put(25, new int[] {784,614,440,330});
        map.put(26, new int[] {842,652,462,365});
        map.put(27, new int[] {902,692,496,385});
        map.put(28, new int[] {940,732,534,405});
        map.put(29, new int[] {1002,778,559,430});
        map.put(30, new int[] {1066,843,604,457});
        map.put(31, new int[] {1132,894,634,486});
        map.put(32, new int[] {1201,947,684,518});
        map.put(33, new int[] {1273,1002,719,553});
        map.put(34, new int[] {1347,1060,756,590});
        map.put(35, new int[] {1417,1113,790,605});
        map.put(36, new int[] {1496,1176,832,647});
        map.put(37, new int[] {1577,1224,876,673});
        map.put(38, new int[] {1661,1292,923,701});
        map.put(39, new int[] {1729,1362,972,750});
        map.put(40, new int[] {1817,1435,1024,784});

        //文字列の長さから最適なVersionを求める
		int     foundVersion = 0;

		for( Integer version = 1; version < 41; version++ ) {
			int datas[] = map.get(version);

			int index = 0;
			if( ecLevel == ErrorCorrectionLevel.L) {
				index = 0;
			}
			else if( ecLevel == ErrorCorrectionLevel.M ) {
				index = 1;
			}
			else if( ecLevel == ErrorCorrectionLevel.Q ) {
				index = 2;
			}
			else if( ecLevel == ErrorCorrectionLevel.H ) {
				index = 3;
			}

			if( datas[index] >= src.length()) {
				foundVersion = version;
				break;
			}
		}

		return foundVersion;
	}

    /**
     * EANパラメータ検査処理<br>
     * EAN画像化できるか検査する<br><br>
     *
     * @param format バーコード指定（BarcodeFormat.EAN_8、EAN_13）
     * @param src 画像化対象文字列
     * @param width 幅
     * @param height 高さ
     * @return True：成功<br>False：失敗
     */
	static boolean checkEANParam(BarcodeFormat format, String src, int width, int height) {
		//幅と高さは0以上である事
		if( width < 0 || height < 0) {
			logout( String.format("画像の幅と高さは0以上で指定してください。width = %d  height = %d", width, height));
        	return false;
        }

		//バーコード指定の範囲検査
		if(format != BarcodeFormat.EAN_13 && format != BarcodeFormat.EAN_8) {
			logout( String.format("バーコードの指定が不正です。"));
			return false;
		}

		if( format == BarcodeFormat.EAN_13) {
			//EAN13なら数字が12または13文字である事
    		if( !src.matches("[0-9]{12,13}") ) {
    			logout( String.format("EAN13で許可されていない文字があるか、長さが不正です。src = %s", src));
    			return false;
    		}
    	}
    	else if( format == BarcodeFormat.EAN_8 )
    	{
    		//EAN8なら数字が7または8文字である事
    		if( !src.matches("[0-9]{7,8}") ) {
    			logout( String.format("EAN8で許可されていない文字があるか、長さが不正です。src = %s", src));
    			return false;
    		}
    	}

		//先頭から国コードを取得する
		int country = Integer.valueOf(src.substring(0, 3));

		//許可されている国コード一覧(FROM - TO のペアごと）
		int countrys[] = {0,9,30,39,50,59,60,139,
				           200, 299,
				           300, 379,
				           380, 380, 383, 383, 385, 385, 387, 387, 389, 389,
				           400, 440,
				           450, 459, 490, 499,
				           460, 469,
				           470, 470, 471, 471, 474, 474, 475, 475, 476, 476, 477, 477, 478, 478, 479, 479,
				           480, 480, 481, 481, 482, 482, 484, 484, 485, 485, 486, 486, 487, 487, 488, 488,
				           489, 489,
				           500, 509,
				           520, 520, 521, 521, 528, 528, 529, 529, 530, 530, 531, 531, 535, 535, 539, 539,
				           540, 549,
				           560, 560, 569, 569,
				           570, 579,
				           590, 590, 594, 594, 599, 599,
				           600, 601,
				           603, 603, 604, 604, 608, 608, 609, 609, 611, 611, 613, 613, 615, 615, 616, 616,
				           618, 618, 619, 619, 621, 621, 622, 622, 624, 624, 625, 625, 626, 626, 627, 627,
				           628, 628, 629, 629,
				           640, 649,
				           690, 695,
				           700, 709,
				           729, 729,
				           730, 739,
				           740, 740, 741, 741, 742, 742, 743, 743, 744, 744, 745, 745, 746, 746, 750, 750,
				           754, 755,
				           759, 759,
				           760, 769,
				           770, 771,
				           773, 773, 775, 775, 777, 777, 779, 779, 780, 780, 784, 784, 786, 786,
				           789, 790,
				           800, 839,
				           840, 849,
				           850, 850, 858, 858, 859, 859, 860, 860, 865, 865, 867, 867,
				           868, 869,
				           870, 879,
				           880, 880, 884, 884, 885, 885, 888, 888, 890, 890, 893, 893, 896, 896, 899, 899,
				           900, 919,
				           930, 939,
				           940, 949,
				           950, 950, 955, 955, 958, 958, 977, 977,
				           978, 979,
				           980, 980, 981, 983, 990, 999 };

		//国コードが許可範囲か検査する
		boolean found = false;
		for( int index = 0; index < countrys.length; index = index + 2) {
			if( countrys[index] <= country && countrys[index+1] >= country) {
				found = true;
				break;
			}
		}
		if( !found ) {
			logout( String.format("先頭3文字の国コードが不正です。country = %d", country));
			return false;
		}

		return true;
	}

    /**
     * NW7チェックデジット付加処理<br>
     * 与えられた文字列にNW7のチェックデジットを付加します。
     * すでにチェックデジットが付加されていた場合は、何もしません。<br><br>
     *
     * @param src 対象文字列
     * @return 付加済み文字列：成功<br>
     */
	static String nw7AppendCheckDegit( String src ) {

		String code = "";
		String orgDegit = "";
		String degit = "*";

		if( src.length() > 3 ) {
			//この長さの時、対象文字にはチェックデジットが付いている

			//先頭からチェックデジットの直前まで(あ)
			//例：「A1234B」なら「A123」
			code = src.substring(0, src.length()-2);

			//現在の文字列のチェツクデジットと思われる文字(い)
			//例：「A1234B」なら「4」
			orgDegit = src.substring(src.length()-2, src.length()-1);

			//(あ)からチェックデジットを計算する
			degit = nw7GetCheckDegit(code);
		} else {
			//先頭からチェックデジットの直前まで(あ)
			//例：「A1B」なら「A1」
			code = src.substring(0, src.length()-1);
		}

		//STOP文字(う)
		//例：「A1234B」なら「B」
		String last = src.substring(src.length()-1, src.length());

		//チェックデジットを付加する
		String retStr = "";

		if( !orgDegit.equals(degit)) {
			// (あ)から得たチェックデジットが、(い)と不一致なら
			// 元の文字列から本来のチェックデジットを計算して、付加して戻す
			degit = nw7GetCheckDegit(code + last);
			retStr = code + degit + last;
		}
		else
		{
			//(あ)から得たチェックデジットが、(い)と一致したので
			//元の文字列はチェックデジット付加済みの文字列と判定
			retStr = src;
		}

		return retStr;
	}

    /**
     * NW7チェックデジット取得処理<br>
     * 与えられた文字列からNW7のチェックデジットを求めます。<br><br>
     *
     * @param src 対象文字列
     * @return チェックデジット：成功<br>
     */
	static String nw7GetCheckDegit(String src) {
		//NW7 文字数値対応：文字と数値のペア
		String codeList[] = { "0","0", "1", "1", "2", "2", "3", "3", "4", "4", "5", "5",
				           "6","6", "7", "7", "8", "8", "9", "9", "-", "10", "$", "11",
				           ":", "12", "/", "13", ".", "14", "+", "15", "A", "16",
				           "B", "17", "C", "18", "D", "19" };

		//与えられた文字列を数値化する
		int sum = 0;
		for( int index = 0; index < src.length(); index++ ) {
			//1文字ずつ取り出す
			String code = src.substring(index, index + 1);
			//数値化する
			for( int i = 0; i < codeList.length; i = i + 2 ) {
				if( code.equals(codeList[i])) {
					//合計する
					sum = sum + Integer.valueOf(codeList[i+1]);
					break;
				}
			}
		}

		//数値化された値から、デジット用数値を求める
		int degitNum = 0;
		int amari = sum % 16;
		if( amari != 0 ) {
			degitNum = 16 - amari;
		}

		//デジット用数値を文字列化する
		String degit = "";
		for( int i = 0; i < codeList.length; i = i + 2 ) {
			if( String.valueOf(degitNum).equals(codeList[i+1])) {
				degit = codeList[i];
				break;
			}
		}

		return degit;
	}

    /**
     * コード化文字付加処理<br>
     * バーコド画像に、コード化した文字を付加します。<br><br>
     *
     * @param app PApplet
     * @param pImage バーコード画像
     * @param src コード化した文字列
     * @param format バーコード形式
     * @param appBar ベアラーバー付加指定（True:付加する  False:付加しない）
     * @return 画像：成功<br>
     */
	static PImage appendCodeImage(PApplet app, PImage pImage, String src, BarcodeFormat format, boolean ...appBar) {

		//文字の大きさを決める
		int fontSize = 1;
		if( pImage.width < pImage.height) {
			fontSize = pImage.height/14;
		}
		else
		{
			fontSize = pImage.width/14;
		}

		String codeText = "";
		String CodeName = "";

		//EAN,NW7はZxing側でチェックデジットを付加させる事があるので、コード化した文字を
		//読み直している
		if( format == BarcodeFormat.EAN_13) {
			CodeName = "EAN13:";
			EANReader reader = new EANReader(app);
			codeText = reader.decode13(pImage);
		}
		else if( format == BarcodeFormat.EAN_8) {
			CodeName = "EAN8:";
			EANReader reader = new EANReader(app);
			codeText = reader.decode8(pImage);
		}
		else if( format == BarcodeFormat.CODABAR) {
			CodeName = "NW7:";
			NW7Reader reader = new NW7Reader(app);
			codeText = reader.decode(pImage);

			//START、STOP、それ以外の文字を取得
			String startTxt = src.substring(0, 1);
			String endTxt = src.substring(src.length()-1, src.length());
			codeText = startTxt + codeText + endTxt;
		}
		else if( format == BarcodeFormat.CODE_39) {
			CodeName = "Code39:";
			codeText = "*" + src + "*";
		}
		else if( format == BarcodeFormat.CODE_128) {
			CodeName = "Code128:";
			codeText = src;
		}
		else if( format == BarcodeFormat.ITF) {
			CodeName = "ITF:";
			codeText = src;
		}

		//ベアラバーを付加する
		if( format == BarcodeFormat.ITF ) {
			boolean addBar = false;
			if( appBar.length == 0 ) {
				addBar = true;
			}
			else {
				addBar = appBar[0];
			}

			if(addBar) {
				//バーの太さを求める
				int barWidth = 0;
				if( pImage.width < pImage.height) {
					barWidth = pImage.width / 12;
				}
				else {
					barWidth = pImage.height / 12;
				}
				if( barWidth < 4 ) { barWidth = 4; }

				PGraphics g = app.createGraphics(pImage.width + barWidth * 2, pImage.height + barWidth * 2);
				g.beginDraw();
				g.background(255);
				g.image( pImage, barWidth, barWidth );
				g.strokeWeight(barWidth);
				g.stroke(0);
				g.noFill();
				g.rect( barWidth/2, barWidth/2, pImage.width + barWidth, pImage.height + barWidth );
				g.endDraw();

				pImage = g.get();
			}
		}

		//文字画像を付加する
		PGraphics g = app.createGraphics(pImage.width, pImage.height + fontSize + 8);
		g.beginDraw();
		g.background(255);
		g.image(pImage,0,0);

		g.textSize(fontSize);
		g.textAlign(PApplet.LEFT, PApplet.TOP);
		g.fill(0);

		int txtStart = (int) ((pImage.width - g.textWidth(CodeName + codeText))/2);
		g.text( CodeName + codeText, txtStart, pImage.height + 8);
		g.endDraw();

		return g.get();
	}

    /**
     * EANチェックデジット削除処理<br>
     * 与えられた文字列から、EAN13またはEAN8のチェックデジットを削除します。
     *
     * @param src 対象文字列
     * @return 付加済み文字列：成功<br>
     */
	static String eanDeleteCheckDegit( String src ) {

		String retStr = "";

		if( src.length() == 13 || src.length() == 8) {
			//チェックデジットを削除する
			retStr = src.substring(0,  src.length()-1);
		}
		else
		{
			//何もしない
			retStr = src;
		}

		return retStr;
	}

    /**
     * NW7パラメータ検査処理<br>
     * NW7画像化できるか検査する<br><br>
     *
     * @param src 画像化対象文字列
     * @param width 幅
     * @param height 高さ
     * @return True：成功<br>False：失敗<br>
     */
	static boolean checkNW7Param(String src, int width, int height) {
		//幅と高さは0以上である事
		if( width < 0 || height < 0) {
			logout( String.format("画像の幅と高さは0以上で指定してください。width = %d  height = %d", width, height));
        	return false;
        }

		//文字列は3文字以上である事（START/STOP + 1 文字）
		if( src.length() < 3) {
			logout( String.format("文字数が不正です src = %s", src));
			return false;
		}

		//START、STOP、それ以外の文字を取得
		String startTxt = src.substring(0, 1);
		String endTxt = src.substring(src.length()-1, src.length());
		String codeTxt = src.substring(1,src.length()-1);

		//START文字はAからDの何れかである事
		if( !startTxt.toUpperCase().matches("[A-D]") ) {
			logout( String.format("START文字がABCDの何れかではありません startTxt = %s", startTxt));
			return false;
		}

		//STOP文字はAからDの何れかである事
		if( !endTxt.toUpperCase().matches("[A-D]") ) {
			logout( String.format("STOP文字がABCDの何れかではありません endTxt = %s", endTxt));
			return false;
		}

		//それ以外の文字は、数字か一部の記号文字である事
		if( !codeTxt.matches("^[0-9$:/.+-]+$") ) {
			logout( String.format("NW7で許可されていない文字があります。src = %s", endTxt));
			return false;
		}

		return true;
	}

	/**
     * ITFパラメータ検査処理<br>
     * ITF画像化できるか検査する<br><br>
     *
     * @param src 画像化対象文字列
     * @param width 幅
     * @param height 高さ
     * @return True：成功<br>False：失敗<br>
	 */
	static boolean checkITFParam(String src, int width, int height) {
		//高さは0以上である事
		if( height < 0) {
			logout( String.format("画像の高さは0以上で指定して下さい。height = %d", height));
        	return false;
        }

		//5,6,13,14,15,16 文字である事
		if( src.length() != 5 && src.length() != 6  ) {
			if( src.length() < 13 && src.length() > 16  ) {
				logout( String.format("文字の長さが不正です。src = %s", src));
	        	return false;
			}
		}

		if( src.length() < 7) {
			if( width < 90 ) {
				logout( String.format("画像の幅は90以上必要です。width = %d", width));
	        	return false;
			}
		}
		else if( src.length() < 15) {
			if( width < 160 ) {
				logout( String.format("画像の幅は160以上必要です。width = %d", width));
	        	return false;
			}
		}
		else {
			if( width < 180 ) {
				logout( String.format("画像の幅は180以上必要です。width = %d", width));
	        	return false;
			}
		}

		//数字のみである事
		if( !src.matches("^[0-9]+$") ) {
			logout( String.format("ITFで許可されていない文字があります。src = %s", src));
			return false;
		}

		return true;
	}

    /**
     * ITFチェックデジット付加処理<br>
     * 与えられた文字列にITFのチェックデジットを付加します。
     * すでにチェックデジットが付加されていた場合は、何もしません。<br><br>
     *
     * @param src 対象文字列
     * @return 付加済み文字列：成功<br>
     */
	static String itfAppendCheckDegit( String src ) {

		String code = src;
		String orgDegit = "";
		String degit = "*";

		if( src.length() == 6 || src.length() == 14 || src.length() == 16 ) {
			//この長さの時、対象文字にはチェックデジットが付いている

			//先頭からチェックデジットの直前まで(あ)
			//例：「012345」なら「01234」
			code = src.substring(0, src.length()-1);

			//現在の文字列のチェツクデジットと思われる文字(い)
			//例：「012345」なら「5」
			orgDegit = src.substring(src.length()-1, src.length());

			//(あ)からチェックデジットを計算する
			degit = itfGetCheckDegit(code);
		}

		//チェックデジットを付加する
		String retStr = "";

		if( !orgDegit.equals(degit)) {
			// (あ)から得たチェックデジットが、(い)と不一致なら
			// 元の文字列から本来のチェックデジットを計算して、付加して戻す
			degit = itfGetCheckDegit(code);
			retStr = code + degit;
		}
		else
		{
			//(あ)から得たチェックデジットが、(い)と一致したので
			//元の文字列はチェックデジット付加済みの文字列と判定
			retStr = src;
		}

		return retStr;
	}

    /**
     * NW7チェックデジット取得処理<br>
     * 与えられた文字列からNW7のチェックデジットを求めます。<br><br>
     *
     * @param src 対象文字列
     * @return チェックデジット：成功<br>
     */
	static String itfGetCheckDegit(String src) {

		//すべての偶数番目の文字を加算する ①
		int gsum = 0;
		for( int index = 0; index < src.length(); index = index + 2) {
			gsum = gsum + Integer.valueOf(src.substring(index, index + 1));
		}

		//①の合計を３倍する　②
		gsum = gsum * 3;

		//すべての奇数番目の文字を加算する ③
		int ksum = 0;
		for( int index = 1; index < src.length(); index = index + 2) {
			ksum = ksum + Integer.valueOf(src.substring(index, index + 1));
		}

		//②と③を加算し、下１桁を求める ④
		int sum = ksum + gsum;
		int lastNum = 0;
		if(sum > 100) {
			lastNum = sum % 100 % 10 ;
		}
		else {
			lastNum = sum % 10 ;
		}

		//④が０でないなら、１０から引いたものがチェックデジット
		int degit = 0;
		if( lastNum > 0) {
			degit = 10 - lastNum;
		}

		return String.valueOf(degit);
	}

    /**
     * Code39チェックデジット付加除外処理<br>
     * 与えられた文字列にCode39のチェックデジットを付加または除外します。<br><br>
     *
     * @param src 対象文字列
     * @param degitFlg チェックデジット付加除外指示（True：付加  False：除外）
     * @return 付加済み文字列：成功<br>
     */
	static String code39AppendDeleteCheckDegit( String src, boolean degitFlg ) {

		//STARTを得る
		//例：「*1234*]なら「*」
		String startTxt = src.substring(0, 1);

		//START記号ありなら、STARTとSTOPを削除する
		if( startTxt.equals("*")) {
			src = src.substring(1, src.length()-1);
		}

		//チェックデジットの手前まで（あ）
		//例：「1234]なら「123」
		String check = src.substring(0, src.length()-1);

		//(あ)からチェックデジットを計算する
		String degit = code39GetCheckDegit(check);

		String retStr = "";
		if(degitFlg) {
			//チェックデジットを付加する場合
			retStr = check + degit;
		}
		else
		{
			//チェックデジットを付加しないなら、元の文字列をそのまま画像化する
			retStr = src;
		}

		return retStr;
	}

    /**
     * Code39チェックデジット取得処理<br>
     * 与えられた文字列からCode39のチェックデジットを求めます。<br><br>
     *
     * @param src 対象文字列
     * @return チェックデジット：成功<br>
     */
	static String code39GetCheckDegit(String src) {
		//文字数値対応：文字と数値のペア
		String codeList[] = { "0","0", "1", "1", "2", "2", "3", "3", "4", "4", "5", "5",
				           "6","6", "7", "7", "8", "8", "9", "9", "A", "10", "B", "11",
				           "C", "12", "D", "13", "E", "14", "F", "15", "G", "16", "H", "17",
				           "I", "18", "J", "19", "K", "20", "L", "21", "M", "22", "N", "23",
				           "O", "24", "P", "25", "Q", "26", "R", "27", "S", "28", "T", "29",
				           "U", "30", "V", "31", "W", "32", "X", "33", "Y", "34", "Z", "35",
				           "-", "36", ".", "37", " ", "38", "$", "39", "/", "40", "+", "41",
				           "%", "42"};

		//START、START・STOPを除いた文字を取得
		String startTxt = src.substring(0, 1);
		String codeTxt = src.substring(1,src.length()-1);

		String checkSrc = "";
		if( startTxt.equals("*")) {
			//START STOP付きなら、それを除外した部分で計算する
			checkSrc = codeTxt;
		} else {
			//START STOPが付いていないので、そのまま計算する
			checkSrc = src;
		}

		//与えられた文字列を数値化する
		int sum = 0;
		for( int index = 0; index < checkSrc.length(); index++ ) {
			//1文字ずつ取り出す
			String code = checkSrc.substring(index, index + 1);
			//数値化する
			for( int i = 0; i < codeList.length; i = i + 2 ) {
				if( code.equals(codeList[i])) {
					//合計する
					sum = sum + Integer.valueOf(codeList[i+1]);
					break;
				}
			}
		}

		//数値化された値から、デジット用数値を求める
		int degitNum = sum % 43;

		//デジット用数値を文字列化する
		String degit = "";
		for( int i = 0; i < codeList.length; i = i + 2 ) {
			if( String.valueOf(degitNum).equals(codeList[i+1])) {
				degit = codeList[i];
				break;
			}
		}

		return degit;
	}

    /**
     * Codeパラメータ検査処理<br>
     * Code39、128画像化できるか検査する<br><br>
     *
     * @param format バーコード指定（BarcodeFormat.CODE_39、CODE_128）
     * @param src 画像化対象文字列
     * @param width 幅
     * @param height 高さ
     * @return True：成功<br>False：失敗
     */
	static boolean checkCodeParam(BarcodeFormat format, String src, int width, int height) {
		//幅と高さは0以上である事
		if( width < 0 || height < 0) {
			logout( String.format("画像の幅と高さは0以上で指定してください。width = %d  height = %d", width, height));
        	return false;
        }

		//バーコード指定の範囲検査
		if(format != BarcodeFormat.CODE_39 && format != BarcodeFormat.CODE_128) {
			logout( String.format("バーコードの指定が不正です。"));
			return false;
		}

		//最低1文字ある事
		if( src.length() < 1) {
			logout( String.format("変換対象文字列が不正です。"));
			return false;
		}

		if(format == BarcodeFormat.CODE_39 ) {
			if( src.length() > 2) {
				//START、STOP 文字を取得
				String startTxt = src.substring(0, 1);
				String endTxt = src.substring(src.length()-1, src.length());

				if(startTxt.equals("*")) {
					if(!endTxt.equals("*")) {
						//START文字ありなら、STOP文字もある事
						logout( String.format("START文字に対応するSTOP文字がありません。src = %s", src));
						return false;
					}
				}
			}

			//Code39、なら、A～Z、数字、一部の記号文字である事
			if( !src.toUpperCase().matches("^[A-Z0-9. $/+%*]+$") ) {
				logout( String.format("Code39で許可されていない文字があります。src = %s", src));
				return false;
			}
		}
		else
		{
			//Code128ならアスキーii文字である事
//			if( !src.toUpperCase().matches("^\\p{ASCII}+$") ) {
//				return false;
//			}
			if( !src.matches("^[\\x20-\\x7F]+$") ) {
				logout( String.format("Code128で許可されていない文字があります。src = %s", src));
				return false;
			}
		}

		return true;
	}

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( ZxingP5Com.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
