package ZxingP5;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.QRCodeWriter;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;

import processing.core.PApplet;
import processing.core.PImage;

//********************************************************
//QR-Code Writer
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Git-Hub に公開されている endoh0509/ZXingP5 をベースに改変
//したものです。
//オリジナルコードのURL : https://github.com/endoh0509/ZXingP5
//
//本APは、参照元となったオリジナルコードの作者：Katsuya Endoh 様に
//敬意を払い、参照元と同じく Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。ただしオリジナルコードについ
//ては、上記 オリジナルコードのURL を参照して下さい。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing QR-Code Writer<br>
 * 本クラスは、指定された文字列をもとに、QR-Codeの画像イメージを作成
 * するものです。<br>
 * 本クラスは、<a href="https://github.com/endoh0509/ZXingP5">Katsuya Endoh 様のオリジナルコード</a>をベースに
 * 作成されています。ここに敬意と感謝を述べさせていただきます。<br><br>
 * 制約事項：<br>
 * ・数字（0～9）、アルファベット、記号、漢字（JIS第一、第二水準）が利用可能<br>
 * ・利用可能な桁数は、利用する文字種、QRバージョン、誤り訂正率により変動する<br>
 * ・QRバージョンを自動計算させる場合は、本ライブラリの制約により、コード化する文字を漢字とみなして計算する<br>
 * ・クワイエットは4ピクセル固定で、QR-Codeにはモデル2が利用される。<br>
 *
 * @see
 * <a href="https://github.com/endoh0509/ZXingP5/">ZXingP5 Original</a><br>
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class QRWriter {

    private QRCodeWriter writer;
    private PImage    rowImage;;
    private BitMatrix bitMatrix;
    private ErrorCorrectionLevel     ecLevel;
    private String    charSet;
    private int	   CodeVersion;
    private PApplet   app;

    public final static String VERSION = "1.0.1";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの書き込み準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRWriter  qrWriter;
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public QRWriter(PApplet applet) {
    	writer = new QRCodeWriter();
    	ecLevel = ErrorCorrectionLevel.M;
    	charSet = "Shift_JIS";
    	CodeVersion = 0;
    	this.app = applet;
    }

    /**
     * 誤り訂正率指定<br>
     * QR-Codeの誤り訂正率を指定します。デフォルトは ErrorCorrectionLevel.M です。<br>
     * Zxingの ErrorCorrectionLevel で指定して下さい。
     * ErrorCorrectionLevelについては <a href="https://zxing.github.io/zxing/apidocs/">公式サイト</a>を参照して下さい。<br><br>
     * @param ec ErrorCorrectionLevel.L 、 M 、 Q 、 H
     * @return True:成功  False:失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRWriter  qrWriter;
     * PImage    pImage;
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //エラー訂正レベルを引き上げる
     *   qrWriter.setEcLevel(ErrorCorrectionLevel.Q);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public boolean setEcLevel(ErrorCorrectionLevel ec) {
    	//範囲検査
    	if( ec != ErrorCorrectionLevel.H &&
    			ec != ErrorCorrectionLevel.L &&
    					ec != ErrorCorrectionLevel.M &&
    							ec != ErrorCorrectionLevel.Q) {

    		ZxingP5Com.logout( String.format("誤り訂正率 引数不正"));
    		return false;
    	}

    	//内部変数を変更
    	ecLevel = ec;
    	return true;
    }

    /**
     * 誤り訂正率取得<br>
     * 現在設定されている誤り訂正率を取得します。<br>
     * 誤り訂正率は、Zxingの ErrorCorrectionLevel で戻されます。<br>
     * ErrorCorrectionLevelについては <a href="https://zxing.github.io/zxing/apidocs/">公式サイト</a>を参照して下さい。<br><br>
     * @return 誤り訂正率:成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * QRWriter    qrWriter;   //QR-Code Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //誤り訂正率取得
     *   ErrorCorrectionLevel ecLevel = qrWriter.getEcLevel();
     *   println( ecLevel );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public ErrorCorrectionLevel getEcLevel() {
    	return ecLevel;
    }

    /**
     * バージョン情報取得<br>
     * QRバージョンを戻します。<br>
     * src を指定すると、QRコード化したい文字列を漢字とみなして、誤り訂正率から最適なバージョン番号を戻します。<br>
     * この場合、もしもQRコード化したい文字列が不適切な場合は、バージョンとして 0 が戻されます。<br><br>
     * src を省略した場合、最後に画像化した文字列のQRバージョンを戻します。この場合、一度もバーコードを画像化していな
     * い場合は、バージョンとして 0 が戻されます。<br><br>
     *
     * @param src QRコード化したい文字列
     * @return QRバージョン(1-40)：成功  0：不正<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * QRWriter    qrWriter;   //QR-Code Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //バージョン取得
     *   int version = qrWriter.getQrVersion("適当なコード化したい文字列");
     *   println( version );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public int getQrVersion(String ...src) {

    	if( src.length != 0) {
    		CodeVersion = ZxingP5Com.getQrVersion(src[0], ecLevel);
    	}
    	return CodeVersion;
    }

    /**
     * バーコード生成<br>
     * 指定された文字列を、指定されたQRバージョンを元にバーコード画像化します。<br>
     * 文字列に使用するキャラクターコードは、QR-Codeの仕様に準拠している必要があります。<br><br>
     * QRバージョンにはQR-Codeで許可されている番号(1-40）を与えます。<br>
     * QRバージョンを省略した場合、QRコード化したい文字列を漢字とみなして、誤り訂正率から最適な
     * バージョン番号を自動計算して画像化します。<br><br>
     * @param src  QRコード化したい文字列
     * @param version QRバージョン(1-40)
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRWriter  qrWriter;  //QR-Code Writer
     * PImage    pImage;    //作成画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //画像化
     *   pImage = qrWriter.encode( "ほげ" );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode(String src, int ...version) {

    	//バージョン未指定なら、最適なバージョンを求める
    	CodeVersion = 0;

    	if( version.length == 0) {
    		CodeVersion = ZxingP5Com.getQrVersion(src, ecLevel);
    		if( CodeVersion == 0) {
    			//文字列が長すぎる
    			ZxingP5Com.logout( String.format("QRバージョン取得失敗 src = %s", src));
    			return null;
    		}
    	} else {
         	//パラメータチェツク
    		CodeVersion = version[0];
        	if( !ZxingP5Com.checkQRParam(src, ecLevel, CodeVersion)) {
        		ZxingP5Com.logout( String.format("パラメータ不正"));
        		return null;
        	}
    	}

        //バージョン情報から、バーコードの大きさを求める
        int size = 21 + CodeVersion * 4;

        //パラメータにバージョンをセットする
    	Map<EncodeHintType, Object> hints = new HashMap<EncodeHintType, Object>();
        hints.put( EncodeHintType.QR_VERSION, CodeVersion );
        hints.put( EncodeHintType.CHARACTER_SET, charSet );
        hints.put( EncodeHintType.ERROR_CORRECTION, ecLevel );
        hints.put( EncodeHintType.MARGIN, 4);

    	//画像化
    	try {
			bitMatrix = writer.encode(src, BarcodeFormat.QR_CODE, size, size, hints);
		} catch (WriterException e) {
			ZxingP5Com.logout( String.format("バーコード生成で例外発生" ));
			return null;
		}

    	//画像（BitMatrix）をPImageに変換する
    	rowImage = ZxingP5Com.changeBitMatrix2PImage( bitMatrix );

    	return( rowImage );
    }

    /**
     * バーコード拡大縮小<br>
     * 画像化したバーコードを、任意の大きさに拡大縮小します。<br>
     * QR-Codeの大きさは encode 時のバージョンにより決定されますが、本メソッドでは
     * これを任意の大きさに変更可能です。最小サイズはwidth、height共に25ピクセルです。<br>
     * ただし指定された大きさがQR-Codeの規約に適合しているかどうかは問われないため、
     * 規約に照らし合わせた場合、不正なものが出来上がる可能性がある事に
     * 注意してください。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRWriter  qrWriter;  //QR-Code Writer
     * PImage    pImage;    //作成画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //画像化
     *   if( qrWriter.encode( "ほげ",1 ) != null ){
     *     //QR画像を拡大
     *     pImage = qrWriter.resize(200,200);
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage resize(int width, int height) {
    	if( width < 25 || height < 25 ) {
    		ZxingP5Com.logout( String.format("画像の高さと幅が不正 width = %d  height = %d", width, height));
    		return null;
    	}

    	PImage pImage = null;
    	if( rowImage != null ) {
    		pImage = ZxingP5Com.resizeImage(rowImage, width, height);
    	}
    	return pImage;
    }

    /**
     * バーコード画像取得<br>
     * 画像化されたイメージを取得します。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRWriter  qrWriter;  //QR-Code Writer
     * PImage    pImage;    //作成画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   qrWriter = new QRWriter();
     *   //画像化
     *   if( qrWriter.encode( "ほげ",1 ) != null ){
     *     //画像取得
     *     pImage = qrWriter.getImage();
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage getImage() {
        return rowImage;
    	//return ZxingP5Com.changeBitMatrix2PImage( this.bitMatrix );
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     * 本ライブラリの<a href="https://github.com/endoh0509/ZXingP5/">参照元となったオリジナルコード</a>の作者：Katsuya Endoh 様
     * に敬意を示し、Versionは1.0.0を欠番としています。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( QRWriter.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}