package ZxingP5;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BinaryBitmap;
import com.google.zxing.ChecksumException;
import com.google.zxing.DecodeHintType;
import com.google.zxing.FormatException;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.qrcode.QRCodeReader;

import processing.core.PApplet;
import processing.core.PImage;
import processing.core.PVector;

//********************************************************
// QR-Code Reader
//------------------------------------------------------------------------
// CopyRight : MSLABO  Ver1.0
//********************************************************
// 本APは Git-Hub に公開されている endoh0509/ZXingP5 をベースに改変
// したものです。
// オリジナルコードのURL : https://github.com/endoh0509/ZXingP5
//
// 本APは、参照元となったオリジナルコードの作者：Katsuya Endoh 様に
// 敬意を払い、参照元と同じく Apache License 2.0 にて提供されます。
//
// 本APの利用にあたっては、必ず自己責任でお願い致します。
//
// 本APの不具合については極力善処を行う予定ですが、これを保証するもの
// ではありません。
// また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
// 原則として個別の問い合わせや要望は受け付けません。
//
// 本APは、下記HPにて情報提供を行います。ただしオリジナルコードについ
// ては、上記 オリジナルコードのURL を参照して下さい。
//
// 本APの情報掲載元：
// URL : http://mslabo.sakura.ne.jp/WordPress/
//
// 本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing QR-Code Reader<br>
 * 本クラスは、QR-Codeが描かれた画像から、画像を解析してコード内容
 * を取得するものです。<br>
 * 本クラスは、<a href="https://github.com/endoh0509/ZXingP5">Katsuya Endoh 様のオリジナルコード</a>をベースに
 * 作成されています。ここに敬意と感謝を述べさせていただきます。<br><br>
 * 制約事項：<br>
 * ・数字（0～9）、アルファベット、記号、漢字（JIS第一、第二水準）が利用可能<br>
 * ・利用可能な桁数は、利用する文字種、QRバージョン、誤り訂正率により変動する<br>
 * ・QRバージョンを自動計算させる場合は、本ライブラリの制約により、コード化する文字を漢字とみなして計算する<br>
 * ・クワイエットは4ピクセル固定で、QR-Codeにはモデル2が利用される。<br>
 *
 * @see
 * <a href="https://github.com/endoh0509/ZXingP5/">ZXingP5 Original</a><br>
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class QRReader {

    private QRCodeReader reader;
    private Result result = null;
    private PApplet app;
    private int rotateDegree;
    public final static String VERSION = "1.0.1";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの読み取り準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     *import ZxingP5.*;
     *QRReader  qrReader;  //QR-Code Reader
     *void setup(){
     *  size(400,400);
     *  //インスタンス作成
     *  qrReader = new QRReader();
     *}
     *void draw(){
     *  background(255);
     *}
     *}
     * </pre>
     */
    public QRReader(PApplet applet) {
        this.reader = new QRCodeReader();
        this.app = applet;
        this.rotateDegree = 10;
    }

	/**
	 * 回転角度指定<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを指定します。<br>
	 * デフォルトは10度です。<br><br>
	 * @param degree 回転角度（1 から 180）
	 */
	public void setRotateDegree(int degree) {
		if( degree > 0 && degree < 181 ) {
			rotateDegree = degree;
		}
	}

	/**
	 * 回転角度取得<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを取得します。<br>
	 * デフォルトは10度です。<br><br>
	 * @return 回転角度（1 から 180）
	 */
	public int getRotateDegree() {
		return rotateDegree;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 本メソッドは内部処理用です。直接呼び出すことはできません。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br><br>
     *
     * @param bitmap ITFが描画された画像
     * @return 解析結果：成功<br>null：失敗<br><br>
     */
	Result decodeRow(BinaryBitmap bitmap) {
		//バーコード読み取り
		this.result = null;

		//解析モードを指定する(精度重視)
        Map<DecodeHintType,Boolean> hints = new HashMap<DecodeHintType, Boolean>();
        hints.put( DecodeHintType.TRY_HARDER, true );

    	//バーコード解析
        try {
            this.result = reader.decode(bitmap);
        } catch (NotFoundException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り NotFoundException 例外発生" ));
            this.result = null;

        } catch (ChecksumException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り ChecksumException 例外発生" ));
            this.result = null;

        } catch (FormatException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り FormatException 例外発生" ));
            this.result = null;
        }


        return this.result;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage QRが描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRReader  qrReader;  //QR-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   qrReader = new QRReader();
     *   pImage = loadImage("qrcodeSample.png");
     *   //画像解析
     *   String text = qrReader.decode( pImage, true );
     *   if( text != null ){
     *     println( "読み取り結果：" + text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage,0,0);
     * }
     * }
     * </pre>
     */
    public String decode(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		PImage copyImage = pImage.copy();

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			decodeRow( bitmap);

	        //成功したら、バーコード文字列を返却する
	        if (this.result != null && this.result.getText() != null) {
	            return this.result.getText();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate( app, pImage, degree );
	        }
	        else {
	        	break;
	        }
		}

        return null;
    }

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 画像にはPImageを渡します。画像はローエンド用ヒストグラムで解析されます。<br><br>
     *
     * @param pImage QRが描画された画像
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRReader  qrReader;  //QR-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   qrReader = new QRReader();
     *   pImage = loadImage("qrcodeSample.png");
     *   //画像解析
     *   String text = qrReader.decode( pImage );
     *   if( text != null ){
     *     println( "読み取り結果：" + text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage,0,0);
     * }
     * }
     * </pre>
     */
    public String decode(PImage pImage) {
        return this.decode(pImage, true, false);
    }

    /**
     * シンボル座標取得<br>
     * バーコード解析結果を使い、シンボル座標を取得します。
     * QRの場合は、3隅の四角い切り出しシンボルと、小さな四角のアライメント座標（モデル2のみ）が得られます。<br>
     * 一度もQRを読み取っていない場合、null が戻されます。<br>
     *
     * @return 座標配列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRReader  qrReader;         //QR-Code Reader
     * PImage    pImage;           //画像
     * PVector   points[] = null;  //シンボル座標
     * void setup(){
     *   size(400,400);
     *   qrReader = new QRReader();
     *   pImage = loadImage("qrcodeSample.png");
     *   //画像解析
     *   String text = qrReader.decode( pImage );
     *   if( text != null ){
     *     println( "読み取り結果：" + text );
     *   }
     *   //シンボル座標取得
     *   points = qrReader.getPoints();
     * }
     * void draw(){
     *   background(255);
     *   image( pImage,0,0);
     *   //シンボル座標に赤い○を描く
     *   fill(color(255,0,0));
     *   if( points != null ){
     *     for( PVector p : points ){
     *       ellipse( p.x, p.y, 10, 10 );
     *     }
     *   }
     * }
     * }
     * </pre>
     */
    public PVector[] getPoints() {
    	return ZxingP5Com.getPoints(this.result);
    }

    /**
     * 解析結果取得<br>
     * 解析結果は、Zxing の Resultクラスのインスタンスとして戻されます。<br>
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">Zxing 公式ドキュメント</a>を参照して
     * ください。<br>
     * 一度もQRを読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 解析結果：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * QRReader  qrReader;         //QR-Code Reader
     * PImage    pImage;           //画像
     * Result    result;           //解析結果
     * void setup(){
     *   size(400,400);
     *   qrReader = new QRReader();
     *   pImage = loadImage("qrcodeSample.png");
     *   //画像解析
     *   qrReader.decode( pImage );
     *   //解析結果取得
     *   result = qrReader.getResult();
     *   if( result != null ){
     *     //読み取り文字表示
     *     println( result.getText() );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage,0,0);
     * }
     * }
     * </pre>
     */
    public Result getResult() {
        return this.result;
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     * 本ライブラリの<a href="https://github.com/endoh0509/ZXingP5/">参照元となったオリジナルコード</a>の作者：Katsuya Endoh 様
     * に敬意を示し、Versionは1.0.0を欠番としています。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( QRReader.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
