package ZxingP5;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.oned.CodaBarWriter;

import processing.core.PApplet;
import processing.core.PImage;

//********************************************************
//NW7-Code Writer
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing NW7-Code Writer<br>
 * 本クラスは、指定された文字列をもとに、NW-7の画像イメージを作成
 * するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * 制約事項：<br>
 * ・数字（0～9）、アルファベット、記号（- , $ , : , / , . , + ）が利用可能<br>
 * ・桁数はスタート/ストップキャラクタを含め3文字以上なら任意<br>
 * ・スタート/ストップキャラクタには ABCDを用いる<br>
 * ・NW7の仕様としてチェックデジットは任意だが、Zxingの制約上必ず付加される<br>
 * ・NW7の仕様としてチェックデジット方式は任意だが、Zxingの制約上モジュラス16が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class NW7Writer {
    private CodaBarWriter writer;
    private PImage rowImage;
    private BitMatrix bitMatrix;
    private PApplet app;

    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの書き込み準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Writer  nwWriter;  //NW7-Code Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   nwWriter = new NW7Writer(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public NW7Writer(PApplet applet) {
        this.writer = new CodaBarWriter();
        this.app = applet;
    }

    /**
     * バーコード拡大縮小<br>
     * 画像化したバーコードを拡大縮小します。<br>
     * 一度もNW7画像を生成していない場合、null が戻されます。<br><br>
     *
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Writer  nwWriter;  //NW7-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   nwWriter = new NW7Writer(this);
     *   //画像化
     *   if( nwWriter.encode( "A1234A", 100, 50 ) != null ){
     *     //NW7画像を拡大
     *     pImage = nwWriter.resize(200,100);
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage resize(int width, int height) {
    	PImage pImage = null;
    	if( rowImage != null ) {
    		pImage = ZxingP5Com.resizeImage(rowImage, width, height);
    	}
    	return pImage;
    }

    /**
     * バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 文字列に使用するキャラクターコードは、NW7-Codeの仕様に準拠している必要があります。<br><br>
     * widthに0を与えた場合や画像化に必要な幅より小さな値の場合、必要な幅は自動的に計算されます。<br><br>
     * appFont に true を指定すると、バーコードの下にコード化した文字列を付加します。デフォルトは「付加しない」です。<br><br>
     * チェックデジットは自動的に付加されます。またチェックデジット付きの文字列が与えられた場合で、かつチェックデジットが
     * 不正な場合は、正しい値に置き換えられます。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅（0以上の値）
     * @param height 画像の高さ（0以上の値）
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Writer  nwWriter;  //NW7-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   nwWriter = new NW7Writer(this);
     *   //画像化
     *   pImage = nwWriter.encode( "A1234A", 200, 100 );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode(String src, int width, int height, boolean ...appFont ) {
    	//パラメータチェツク
        if( !ZxingP5Com.checkNW7Param( src, width, height)) {
        	ZxingP5Com.logout( String.format("パラメータ不正"));
        	return null;
        }

        //画像化したい文字列にチェックデジットを付加する
        String targer = ZxingP5Com.nw7AppendCheckDegit(src.toUpperCase());

        //画像化
    	try {
    		bitMatrix = this.writer.encode(targer.toUpperCase(), BarcodeFormat.CODABAR, width, height);
        } catch (WriterException e) {
        	ZxingP5Com.logout( String.format("バーコード生成で例外発生" ));
        	return null;
        }

    	//画像（BitMatrix）をPImageに変換する
    	rowImage  = ZxingP5Com.changeBitMatrix2PImage(bitMatrix);

    	//文字を付加する
    	PImage pImage = rowImage.copy();
    	if( appFont.length > 0 && appFont[0] == true ) {
    		pImage = ZxingP5Com.appendCodeImage(app, rowImage, targer.toUpperCase(), BarcodeFormat.CODABAR);
    	}

    	return(pImage);
    }

    /**
     * バーコード画像取得<br>
     * 画像化されたイメージを取得します。<br>
     * 本メソッドで戻される画像イメージには、コード化された文字画像は含まれていません。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Writer  nwWriter;  //NW7-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   nwWriter = new NW7Writer(this);
     *   //画像化
     *   if( nwWriter.encode( "A1234A", 200, 100 ) != null ){
     *     //NW7画像を取得
     *     pImage = nwWriter.getImage();
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
	public PImage getImage() {
		return rowImage;
    	//return ZxingP5Com.changeBitMatrix2PImage( bitMatrix );
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( NW7Writer.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
