package ZxingP5;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.FormatException;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.oned.CodaBarReader;

import processing.core.PApplet;
import processing.core.PImage;
import processing.core.PVector;

//********************************************************
//NW7-Code Reader
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing NW7-Code Reader<br>
 * 本クラスは、NW-7が描かれた画像から、画像を解析してコード内容
 * を取得するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * 制約事項：<br>
 * ・数字（0～9）、アルファベット、記号（- , $ , : , / , . , + ）が利用可能<br>
 * ・桁数はスタート/ストップキャラクタを含め3文字以上なら任意<br>
 * ・スタート/ストップキャラクタには ABCDを用いる<br>
 * ・NW7の仕様としてチェックデジットは任意だが、Zxingの制約上必ず付加される<br>
 * ・NW7の仕様としてチェックデジット方式は任意だが、Zxingの制約上モジュラス16が使われる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class NW7Reader {
	private CodaBarReader reader;
	private Result result = null;
	private PApplet app;
    private int rotateDegree;
    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの読み取り準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Reader nwReader;  //NW7-Code Reader
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   nwReader = new NW7Reader(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
	public NW7Reader(PApplet applet) {
        this.reader = new CodaBarReader();
        this.app = applet;
        this.rotateDegree = 10;
    }

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 本メソッドは内部処理用です。直接呼び出すことはできません。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br><br>
     *
     * @param bitmap NW7が描画された画像
     * @return 解析結果：成功<br>null：失敗<br><br>
     */
	Result decodeRow( BinaryBitmap bitmap) {
		this.result = null;

		//解析モードを指定する(精度重視)
        Map<DecodeHintType,Object> hints = new HashMap<DecodeHintType, Object>();
        hints.put( DecodeHintType.TRY_HARDER, true );

        //バーコード解析
        try {
        	this.result = reader.decode(bitmap, hints);

        } catch (NotFoundException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り NotFoundException 例外発生" ));
        } catch (FormatException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り FormatException 例外発生" ));
        }

        return this.result;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * スタート／ストップキャラクタは読み取り文字に反映されません。<br>
     * また読み取り結果はチェックデジット文字付きで戻されます。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage NW7が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Reader nwReader;  //NW7-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("nw7Sample.png");
     *   //インスタンス作成
     *   nwReader = new NW7Reader();
     *   //画像解析
     *   String text = nwReader.decode( pImage, true );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		PImage copyImage = pImage.copy();

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			decodeRow( bitmap);

	        if (this.result != null && this.result.getText() != null) {
	        	//成功
	            return this.result.getText();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate( app, pImage, degree );
	        }
	        else {
	        	break;
	        }
		}

        return null;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。画像にはPImageを渡します。<br>
     * スタート／ストップキャラクタは読み取り文字に反映されません。<br>
     * また読み取り結果はチェックデジット文字付きで戻されます。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br><br>
     *
     * @param pImage NW7が描画された画像
     * @return バーコード文字列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Reader nwReader;  //NW7-Code Reader
     * PImage    pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("nw7Sample.png");
     *   //インスタンス作成
     *   nwReader = new NW7Reader();
     *   //画像解析
     *   String text = nwReader.decode( pImage );
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode(PImage pImage) {
        return this.decode(pImage, true, false);
    }

    /**
     * 頂点座標取得<br>
     * バーコード解析結果を使い、頂点座標を取得します。<br>
     * NW7の場合は、バーコードの左端と右端のセンター座標が得られます。一度もNW7を読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 座標配列：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Reader nwReader;  //NW7-Code Reader
     * PImage    pImage;    //画像
     * PVector[]  points;    //頂点座標
     * void setup(){
     *   size(400,400);
     *   pImage = loadImage("nw7Sample.png");
     *   //インスタンス作成
     *   nwReader = new NW7Reader();
     *   //画像解析
     *   nwReader.decode( pImage );
     *   //頂点座標取得
     *   points = nwReader.getPoints();
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     *
     *   //シンボル座標に赤い○を描く
     *   fill(color(255,0,0));
     *   if( points != null ){
     *     for( PVector p : points ){
     *       ellipse( p.x, p.y, 10, 10 );
     *     }
     *   }
     * }
     * }
     * </pre>
     */
    public PVector[] getPoints() {
    	return ZxingP5Com.getPoints(this.result);
  	}

    /**
     * 解析結果取得<br>
     * バーコード解析結果を取得します。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br>
     * 一度もNW7を読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 解析結果：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * NW7Reader nwReader;       //NW7-Code Reader
     * PImage    pImage;         //画像
     * Result    result;         //解析結果
     * void setup(){
     *   size(400,400);
     *   nwReader = new NW7Reader();
     *   pImage = loadImage("nw7Sample.png");
     *   //画像解析
     *   nwReader.decode( pImage );
     *   //解析結果取得
     *   result = nwReader.getResult();
     *   if( result != null ){
     *     //読み取り文字表示
     *     println( result.getText() );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage,0,0);
     * }
     * }
     * </pre>
     */
    public Result getResult() {
        return this.result;
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( NW7Reader.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
