package ZxingP5;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.oned.ITFWriter;

import processing.core.PApplet;
import processing.core.PImage;

//********************************************************
//ITF-Code Writer
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************
/**
 * PROCESSING Zxing ITF-Code Writer<br>
 * 本クラスは、指定された文字列をもとに、ITFの画像イメージを作成
 * するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * 制約事項：<br>
 * ・数字（0～9）が利用可能<br>
 * ・桁数はチェックデジット込みで、6文字、14文字または16文字固定<br>
 * ・チェックデジットは必須で、かならず付加される<br>
 * ・チェックデジット方式にはモジュラス10ウェイト3方式が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class ITFP5Writer {
    private ITFWriter writer;
    private PImage rowImage;
    private BitMatrix bitMatrix;
    private PApplet app;

    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの書き込み準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * ITFP5Writer  itfWriter;  //ITF-Code Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   itfWriter = new ITFP5Writer(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public ITFP5Writer(PApplet applet) {
        this.writer = new ITFWriter();
        this.app = applet;
    }

    /**
     * バーコード拡大縮小<br>
     * 画像化したバーコードを拡大縮小します。<br>
     * 一度もITF画像を生成していない場合、null が戻されます。<br><br>
     *
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * ITFP5Writer  itfWriter;  //ITF-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   itfWriter = new ITFP5Writer(this);
     *   //画像化
     *   if( itfWriter.encode( "0123456789012", 100, 50 ) != null ){
     *     //画像を拡大
     *     pImage = itfWriter.resize(200,100);
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage resize(int width, int height) {
    	PImage pImage = null;
    	if( rowImage != null ) {
    		pImage = ZxingP5Com.resizeImage(rowImage, width, height);
    	}
    	return pImage;
    }

    /**
     * バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 文字列に使用するキャラクターコードは、ITF-Codeの仕様に準拠している必要があります。<br><br>
     * widthに0を与えた場合や画像化に必要な幅より小さな値の場合、必要な幅は自動的に計算されます。<br><br>
     * appFont に true を指定すると、バーコードの下にコード化した文字列を付加します。デフォルトは「付加しない」です。<br><br>
     * appBar に true を指定すると、バーコードの周りにベアラーバーを付加します。デフォルトは「付加する」です。<br><br>
     * チェックデジットは自動的に付加されます。またチェックデジット付きの文字列が与えられた場合で、かつチェックデジットが
     * 不正な場合は、正しい値に置き換えられます。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅（6文字なら90以上、14文字なら160以上、16文字なら180以上）
     * @param height 画像の高さ（0以上の値）
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @param appBar ベアラーバー付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * ITFP5Writer  itfWriter;  //ITF-Code Writer
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   itfWriter = new ITFP5Writer(this);
     *   //画像化
     *   pImage = itfWriter.encode( "0123456789012", 200, 100, true, true );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode(String src, int width, int height, boolean appFont, boolean ...appBar ) {
    	//パラメータチェツク
        if( !ZxingP5Com.checkITFParam( src, width, height)) {
        	ZxingP5Com.logout( String.format("パラメータ不正"));
        	return null;
        }

        //画像化したい文字列にチェックデジットを付加する
        String targer = ZxingP5Com.itfAppendCheckDegit(src.toUpperCase());

        //画像化
    	try {
    		bitMatrix = this.writer.encode(targer, BarcodeFormat.ITF, width, height);
        } catch (WriterException e) {
        	ZxingP5Com.logout( String.format("バーコード生成で WriterException 例外発生" ));
        	return null;
        } catch( IllegalArgumentException e) {
	    	ZxingP5Com.logout( String.format("バーコード生成で IllegalArgumentException 例外発生" ));
	    	return null;
	    }

    	//画像（BitMatrix）をPImageに変換する
    	rowImage  = ZxingP5Com.changeBitMatrix2PImage(bitMatrix);

    	//文字を付加する
    	PImage pImage = rowImage.copy();
    	if( appFont == true ) {
    		pImage = ZxingP5Com.appendCodeImage(app, rowImage, targer.toUpperCase(), BarcodeFormat.ITF, appBar);
    	}

    	return(pImage);
    }

    /**
     * バーコード画像取得<br>
     * 画像化されたイメージを取得します。<br>
     * 本メソッドで戻される画像イメージには、コード化された文字画像は含まれていません。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * ITFP5Writer  itfWriter;  //ITF-Code Writerr
     * PImage     pImage;    //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   itfWriter = new ITFP5Writer(this);
     *   //画像化
     *   if( itfWriter.encode( "01234", 200, 100, true, true ) != null ){
     *     //ITF画像を取得
     *     pImage = itfWriter.getImage();
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
	public PImage getImage() {
		return rowImage;
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( ITFP5Writer.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
