package ZxingP5;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.oned.Code128Writer;
import com.google.zxing.oned.Code39Writer;

import processing.core.PApplet;
import processing.core.PImage;

//********************************************************
//Code39、Code93、Code128 Writer
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing Code39、Code128 Writer<br>
 * 本クラスは、指定された文字列をもとに、Code39、Code128の画像イメージ
 * を作成するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * Code39制約事項：<br>
 * ・数値（0～9）、アルファベット、記号（— , . , スペース, *, $ , /, + , %）が利用可能<br>
 * ・桁数は1文字以上なら任意<br>
 * ・スタート/ストップキャラクタには * を用いる<br>
 * ・仕様としてチェックデジットの付加は任意<br>
 * ・チェックデジット方式にはモジュラス43が用いられる<br><br>
 * Code128制約事項：<br>
 * ・ASCiiコード（0x20～0x7F）が利用可能<br>
 * ・桁数は1文字以上なら任意<br>
 * ・仕様としてCode-A、B、C があるが、Zxingの制約により Code-A のみ書き込み可能<br>
 * ・画像化する際にはスタートコードを除いた文字列を与える事<br>
 * ・チェックデジットは必須。画像化する際、チェックデジットはZxing側で自動付加される<br>
 * ・チェックデジット方式には、重み付きのモジュラース103が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class CodeWriter {
    private Code39Writer writer39;
    private Code128Writer writer128;
    private PImage rowImage;;
    private BitMatrix bitMatrix;
    private PApplet app;

    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの書き込み準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter;  //Code39_128 Writer
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public CodeWriter(PApplet applet) {
        this.writer39 = new Code39Writer();
        this.writer128 = new Code128Writer();
        this.app = applet;
    }

    /**
     * バーコード拡大縮小<br>
     * 画像化したバーコードを拡大縮小します。<br>
     * 一度も画像を生成していない場合、null が戻されます。<br><br>
     *
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     *   //画像化
     *   String code = "AB123XY";
     *   if( codeWriter.encode39( code, 150, 50, false ) != null ){
     *     pImage = codeWriter.resize(350, 50);
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage resize(int width, int height) {
    	PImage pImage = null;
    	if( rowImage != null ) {
    		pImage = ZxingP5Com.resizeImage(rowImage, width, height);
    	}
    	return pImage;
    }

    /**
     * バーコード生成<br>
     * 指定された文字列を、指定された大きさでバーコード画像化します。<br>
     * 文字列に使用するキャラクターコードは、Code39またはCode128の仕様に準拠している必要があります。<br>
     * ただし、Code128は必ずCode-Aとして作成されます。<br><br>
     * widthに0を与えた場合や画像化に必要な幅より小さな値の場合、必要な幅は自動的に計算されます。<br>
     * appFont に true を指定すると、バーコードの下にコード化した文字列を付加します。デフォルトは「付加しない」です。<br><br>
     * Code39にSTART・STOP記号が付加されていない場合は、自動で付加されます。<br>
     * またCode39の場合は、appDegit でチェックデジットを付加するか否かを制御できます。<br>
     * Code39で appDegit に true を指定した場合、与えた文字列のSTOP記号を除く最終文字をチェックデジットと
     * みなし、それを削除した上で、ライブラリ側で自動計算したチェックデジットを付加し直します。<br><br>
     * Code128化したい文字列にはSTARTコード、チェックデジットを除外した文字列を与えてください。<br>
     * Code128の場合、STARTコード、チェックデジットは、appDegit の指定に関わらず、ライブラリ側で強制的に
     * 付加されます。<br><br>
     *
     * @param src 画像化したい文字列
     * @param format バーコード指定（BarcodeFormat.CODE_39、CODE_128）
     * @param width 画像の横幅（0以上の値）
     * @param height 画像の高さ（0以上の値）
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @param appDegit Code39チェックデジット付加除外指示（True：付加する  False：除外する）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     *   //画像化
     *   pImage = codeWriter.encode( "AB123XY", BarcodeFormat.CODE_39, 350, 50, true, false );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode(String src, BarcodeFormat format, int width, int height, boolean appFont, boolean ...appDegit ) {
    	//パラメータチェツク
        if( !ZxingP5Com.checkCodeParam(format, src, width, height)) {
        	ZxingP5Com.logout( String.format("パラメータ不正"));
        	return null;
        }

        //画像化
    	try {
        	if( format == BarcodeFormat.CODE_39) {
        		//STARTを得る
        		String startTxt = src.substring(0, 1);

        		//START記号ありなら、STARTとSTOPを削除する
        		if( startTxt.equals("*")) {
        			src = src.substring(1, src.length()-1);
        		}

        		//チェックデジットを付加または除外する
                if( appDegit.length == 0 ) {
                	//除外する
                	src = ZxingP5Com.code39AppendDeleteCheckDegit(src.toUpperCase(), false);
                }
                else
                {
                	//指定に従う
                	src = ZxingP5Com.code39AppendDeleteCheckDegit(src.toUpperCase(), appDegit[0]);
                }


        		bitMatrix = this.writer39.encode(src.toUpperCase(), format, width, height);
        	}
        	else
        	{
        		bitMatrix = this.writer128.encode(src, format, width, height);
        	}
        } catch (WriterException e) {
        	ZxingP5Com.logout( String.format("バーコード生成で WriterException 例外発生" ));
        	return null;
	    } catch( IllegalArgumentException e) {
	    	ZxingP5Com.logout( String.format("バーコード生成で IllegalArgumentException 例外発生" ));
	    	return null;
	    }

    	//画像（BitMatrix）をPImageに変換する
    	rowImage  = ZxingP5Com.changeBitMatrix2PImage(bitMatrix);

    	//コード化した文字を付加する
    	PImage pImage = rowImage.copy();
    	if( appFont ) {
    		pImage = ZxingP5Com.appendCodeImage(app, rowImage, src, format);
    	}

    	return(pImage);
    }

    /**
     * Code128 バーコード生成<br>
     * 指定された文字列を、指定された大きさでCode128として画像化します。<br>
     * 詳細は encode を参照して下さい。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     *   //画像化
     *   String code = "AB123XY";
     *   pImage = codeWriter.encode128( code, 350, 50, false );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode128(String src,  int width, int height, boolean appFont ){
    	return encode(src, BarcodeFormat.CODE_128, width, height, appFont, false );
    }

    /**
     * Code39 バーコード生成<br>
     * 指定された文字列を、指定された大きさでCode39として画像化します。<br>
     * 詳細は encode を参照して下さい。<br><br>
     *
     * @param src 画像化したい文字列
     * @param width 画像の横幅
     * @param height 画像の高さ
     * @param appFont バーコード文字付加指定（True:付加する  False:付加しない）
     * @param appDegit Code39チェックデジット付加除外指示（True：付加  False：除外）
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     *   //画像化
     *   String code = "AB123XY";
     *   pImage = pImage = codeWriter.encode39( code, 350, 50, false, false );
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
    public PImage encode39(String src,  int width, int height, boolean appFont, boolean ...appDegit){
    	return encode(src, BarcodeFormat.CODE_39, width, height, appFont, appDegit);
    }

    /**
     * バーコード画像取得<br>
     * 画像化されたイメージを取得します。<br>
     * 本メソッドで戻される画像イメージには、コード化された文字画像は含まれていません。<br>
     * 一度もバーコードを画像化していない場合、null が戻されます。<br><br>
     *
     * @return バーコード画像：成功<br>null：失敗<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeWriter  codeWriter; //Code39_128 Writer
     * PImage      pImage;     //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeWriter = new CodeWriter(this);
     *   //画像化
     *   String code = "AB123XY";
     *   if( codeWriter.encode39( code, 350, 50, false ) != null ){
     *     pImage = codeWriter.getImage();
     *   }
     * }
     * void draw(){
     *   background(255);
     *   if( pImage != null ){
     *     image( pImage, 0, 0 );
     *   }
     * }
     * }
     * </pre>
     */
	public PImage getImage() {
		return rowImage;
    	//return ZxingP5Com.changeBitMatrix2PImage( bitMatrix);
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( CodeWriter.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
