package ZxingP5;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.FormatException;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.oned.Code128Reader;
import com.google.zxing.oned.Code39Reader;

import processing.core.PApplet;
import processing.core.PImage;
import processing.core.PVector;

//********************************************************
//Code39、Code93、Code128 Reader
//------------------------------------------------------------------------
//CopyRight : MSLABO  Ver1.0
//********************************************************
//本APは Apache License 2.0 にて提供されます。
//
//本APの利用にあたっては、必ず自己責任でお願い致します。
//
//本APの不具合については極力善処を行う予定ですが、これを保証するもの
//ではありません。
//また本APに関する情報はすべてMSLABOのHPから一方的に公開するものとし
//原則として個別の問い合わせや要望は受け付けません。
//
//本APは、下記HPにて情報提供を行います。
//
//本APの情報掲載元：
//URL : http://mslabo.sakura.ne.jp/WordPress/
//
//本APが、みなさまのPROCESSING利用の一助になれば幸いです。
//
//***********************************************************

/**
 * PROCESSING Zxing Code39、Code128 Reader<br>
 * 本クラスは、Code39、Code128が描かれた画像から、画像を解析して
 * コード内容を取得するものです。<br>
 * 本クラスは MSLABO のオリジナルコードです。<br><br>
 * Code39制約事項：<br>
 * ・数値（0～9）、アルファベット、記号（— , . , スペース, *, $ , /, + , %）が利用可能<br>
 * ・桁数は1文字以上なら任意<br>
 * ・スタート/ストップキャラクタには * を用いる<br>
 * ・仕様としてチェックデジットの付加は任意<br>
 * ・チェックデジット方式にはモジュラス43が用いられる<br><br>
 * Code128制約事項：<br>
 * ・ASCiiコード（0x20～0x7F）が利用可能<br>
 * ・桁数は1文字以上なら任意<br>
 * ・仕様としてCode-A、B、C があるが、Zxingの制約により Code-A のみ書き込み可能<br>
 * ・画像化する際にはスタートコードを除いた文字列を与える事<br>
 * ・チェックデジットは必須。画像化する際、チェックデジットはZxing側で自動付加される<br>
 * ・チェックデジット方式には、重み付きのモジュラース103が用いられる<br><br>
 *
 * @see
 * <a href="https://github.com/zxing/zxing/">zxing</a><br>
 * <a href="http://mslabo.sakura.ne.jp/WordPress/">本APの情報公開元</a>
 * <br><br>
 *
 * @author MSLABO
 * @version 2018/05  1.0
 */
public class CodeReader {
	private Code39Reader reader39;
	private Code128Reader  reader128;
	private Result result = null;
	private PApplet app;
    private int rotateDegree;
    public final static String VERSION = "1.0.0";

    /**
     * コンストラクタ<br>
     * クラスを初期化し、バーコードの読み取り準備を行います。<br><br>
     *
     * @param applet PROCESSINGアプレット<br><br>
     *
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeReader codeReader;  //Code39_128 Reader
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader(this);
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
	public CodeReader(PApplet applet) {
        this.reader39 = new Code39Reader();
        this.reader128 = new Code128Reader();
        this.app = applet;
        this.rotateDegree = 10;
    }

	/**
	 * 回転角度指定<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを指定します。<br>
	 * デフォルトは10度です。<br><br>
	 * @param degree 回転角度（1 から 180）
	 */
	public void setRotateDegree(int degree) {
		if( degree > 0 && degree < 181 ) {
			rotateDegree = degree;
		}
	}

	/**
	 * 回転角度取得<br>
	 * decodeで回転読み取りを指定された場合、何度回転するかを取得します。<br>
	 * デフォルトは10度です。<br><br>
	 * @return 回転角度（1 から 180）
	 */
	public int getRotateDegree() {
		return rotateDegree;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 本メソッドは内部処理用です。直接呼び出すことはできません。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。<br>
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br><br>
     *
     * @param format バーコード指定（BarcodeFormat.CODE_39、CODE_128）
     * @param bitmap Code39またはCode128が描画された画像
     * @return 解析結果：成功<br>null：失敗
     */
	Result decodeRow(BarcodeFormat format, BinaryBitmap bitmap) {
		this.result = null;

		//解析モードを指定する(精度重視)
        Map<DecodeHintType,Boolean> hints = new HashMap<DecodeHintType, Boolean>();
        hints.put( DecodeHintType.TRY_HARDER, true );

        //バーコード解析
        try {
        	if( format == BarcodeFormat.CODE_39) {
        		this.result = reader39.decode(bitmap, hints);
        	}
        	else {
        		this.result = reader128.decode(bitmap, hints);
        	}

        } catch (NotFoundException e) {
    		ZxingP5Com.logout( String.format("バーコード読み取り NotFoundException 例外発生"));
        } catch (FormatException e) {
        	ZxingP5Com.logout( String.format("バーコード読み取り FormatException 例外発生"));
        }

        return this.result;
	}

    /**
     * Code39 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取り結果は、文字列として戻されます。<br>
     * 詳しくは decode39Byte を参照して下さい。<br>
     *
     * @param pImage Code39が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code39Sample.png");
     *   //画像解析
     *   String text = codeReader.decode39(pImage, true);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode39(PImage pImage, boolean globalHistogram, boolean ...doRotate) {
		//バーコード解析
		decode39Byte( pImage, globalHistogram, doRotate );

        if (this.result != null ) {
            return this.result.getText();
        }
        return null;
	}

    /**
     * Code39 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 読み取り結果は、文字列として戻されます。<br>
     * 詳しくは decode39Byte を参照して下さい。<br><br>
     *
     * @param pImage Code39が描画された画像
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code39Sample.png");
     *   //画像解析
     *   String text = codeReader.decode39(pImage);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode39(PImage pImage) {
        return this.decode39(pImage, true, false);
    }

    /**
     * Code39 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取る対象画像が、あらかじめ Code39 だとわかっている場合に利用します。<br>
     * 画像にはPImageを渡します。読み取り結果は、バイト配列として戻されます。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage Code39が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code39Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decode39Byte(pImage, true);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public byte[] decode39Byte(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		PImage copyImage = pImage.copy();

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			//バーコード解析
			decodeRow(BarcodeFormat.CODE_39, bitmap);

	        if (this.result != null ) {
	            return this.result.getText().getBytes();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate( app, pImage, degree );
	        }
	        else {
	        	break;
	        }
		}

        return null;
	}

    /**
     * Code39 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取り結果は、バイト配列として戻されます。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 詳しくは decode39Byte を参照して下さい。<br><br>
     *
     * @param pImage Code39が描画された画像
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code39Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decode39Byte(pImage);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public byte[]  decode39Byte(PImage pImage) {
        return this.decode39Byte(pImage, true, false);
    }

    /**
     * Code128 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取り結果は、文字列として戻されます。<br>
     * 詳しくは decode128Byte を参照して下さい。<br><br>
     *
     * @param pImage Code128が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   String text = codeReader.decode128(pImage, true);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode128(PImage pImage, boolean globalHistogram, boolean ...doRotate) {
		//バーコード解析
		decode128Byte( pImage, globalHistogram, doRotate );

        if (this.result != null ) {
            return this.result.getText();
        }
        return null;
	}

    /**
     * Code128 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取り結果は、文字列として戻されます。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 詳しくは decode128Byte を参照して下さい。<br><br>
     *
     * @param pImage Code128が描画された画像
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   String text = codeReader.decode128(pImage);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode128(PImage pImage) {
        return this.decode128(pImage, true, false);
    }

    /**
     * Code128 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取る対象画像が、あらかじめ Code128 だとわかっている場合に利用します。<br>
     * 読み取り結果は、バイト配列として戻されます。<br><br>
     * doRotateにtrueを指定すると、画像を0度から180度の間で回転しながら読み取ります。デフォルトは
     * 「回転しない」です。<br><br>
     *
     * @param pImage Code128が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decode128Byte(pImage, true);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public byte[] decode128Byte(PImage pImage, boolean globalHistogram, boolean ...doRotate) {

		PImage copyImage = pImage.copy();

		for( int degree = 0; degree <= 180; degree += this.rotateDegree) {
			//PImageをBinaryBitmapに変換
			BinaryBitmap bitmap = ZxingP5Com.changePImage2Bitmap( copyImage, globalHistogram );

			//バーコード読み取り
			decodeRow(BarcodeFormat.CODE_128, bitmap);

			if (this.result != null ) {
	            return this.result.getRawBytes();
	        }

	        if( doRotate.length > 0 && doRotate[0] == true ) {
	        	ZxingP5Com.logout( String.format("画像を %d 度回転して、再度読み取ります", degree ));
	        	copyImage = ZxingP5Com.codeImageRotate( app, pImage, degree );
	        }
	        else {
	        	break;
	        }
		}

        return null;
	}

    /**
     * Code128 バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 読み取り結果は、バイト配列として戻されます。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 詳しくは decode128Byte を参照して下さい。<br><br>
     *
     * @param pImage Code128が描画された画像
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decode128Byte(pImage);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public byte[]  decode128Byte(PImage pImage) {
        return this.decode128Byte(pImage, true, false);
    }

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * まず始めに画像をCode39として読み取り、NGの場合はCode128として読み取ります。<br>
     * 読み取り結果は、文字列として戻されます。ただし Code128 に特殊文字が含まれていた場合は
     * 該当文字が化ける可能性がある事に注意してください。<br><br>
     *
     * @param pImage Code39またはCode128が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   String text = codeReader.decode(pImage, true);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public String decode(PImage pImage, boolean globalHistogram, boolean ...doRotate) {
		//Code39として読み取ってみる
        String retText = decode39(pImage, globalHistogram, doRotate);
        if( retText == null) {
        	//NG なら Code128 として読み取ってみる
        	ZxingP5Com.logout("Code39では無さそうです。Code128として読み取ります。");

        	retText = decode128(pImage, globalHistogram, doRotate);
        }

        return retText;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 詳しくは decode を参照して下さい。<br><br>
     *
     * @param pImage Code39またはCode128が描画された画像
     * @return バーコード文字列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   String text = codeReader.decode(pImage);
     *   if( text != null ){
     *     println( text );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public String decode(PImage pImage) {
        return this.decode(pImage, true, false);
    }

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * まず始めに画像をCode39として読み取り、NGの場合はCode128として読み取ります。<br>
     * 読み取り結果は、バイト配列として戻されます。<br><br>
     *
     * @param pImage Code39またはCode128が描画された画像
     * @param globalHistogram ヒストグラム指定 True：ローエンド用ヒストグラム False：汎用ヒストグラム
     * @param doRotate 回転指示  True 回転して読み取る
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decodeByte(pImage, true);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
	public byte[] decodeByte(PImage pImage, boolean globalHistogram, boolean ...doRotate) {
		//Code39として読み取ってみる
        byte[] retByte = decode39Byte(pImage, globalHistogram, doRotate);
        if( retByte == null) {
        	//NG なら Code128 として読み取ってみる
        	ZxingP5Com.logout( String.format("Code39では無さそうです。Code128として読み取ります。"));

        	retByte = decode128Byte(pImage, globalHistogram, doRotate);
        }

        return retByte;
	}

    /**
     * バーコード読み取り<br>
     * 指定された画像からバーコードを読み取ります。<br>
     * 画像はローエンド用ヒストグラムで解析されます。<br>
     * 詳しくは decodeByte を参照して下さい。<br><br>
     *
     * @param pImage Code39またはCode128が描画された画像
     * @return バイト配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   byte[] bits = codeReader.decodeByte(pImage);
     *   if( bits != null ){
     *     println( bits );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public byte[] decodeByte(PImage pImage) {
        return this.decodeByte(pImage, true);
    }

    /**
     * 頂点座標取得<br>
     * バーコード解析結果を使い、頂点座標を取得します。<br>
     * Code39、Code128の場合は、バーコードの左端と右端のセンター座標が得られます。<br>
     * 一度も画像を読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 座標配列：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * PVector[]  points;
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   codeReader.decode128Text(pImage);
     *   //頂点座標取得
     *   points = codeReader.getPoints();
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     *   //シンボル座標に赤い○を描く
     *     fill(color(255,0,0));
     *     if( points != null ){
     *       for( PVector p : points ){
     *         ellipse( p.x, p.y, 10, 10 );
     *       }
     *     }
     *   }
     * }
     * </pre>
     */
    public PVector[] getPoints() {
    	return ZxingP5Com.getPoints(this.result);
  	}

    /**
     * 解析結果取得<br>
     * バーコード解析結果を取得します。<br>
     * 解析結果は、Resultクラスのインスタンスとして戻されます。<br>
     * Resultクラスについては、<a href="https://zxing.github.io/zxing/apidocs/">zxing 公式ドキュメント</a>を参照して
     * ください。<br>
     * 一度も画像を読み取っていない場合、null が戻されます。<br><br>
     *
     * @return 解析結果：成功<br>null：失敗
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * CodeReader codeReader;  //Code39_128 Reader
     * PImage     pImage;      //画像
     * Result    result;       //解析結果
     * void setup(){
     *   size(400,400);
     *   //インスタンス作成
     *   codeReader = new CodeReader();
     *   pImage = loadImage("code128Sample.png");
     *   //画像解析
     *   codeReader.decode128Text(pImage);
     *   //解析結果取得
     *   result = codeReader.getResult();
     *   if( result != null ){
     *     println( result.getText() );
     *   }
     * }
     * void draw(){
     *   background(255);
     *   image( pImage, 0, 0 );
     * }
     * }
     * </pre>
     */
    public Result getResult() {
        return this.result;
    }

    /**
     * バージョン取得<br>
     * 本ライブラリのバーションを取得します。<br>
     *
     * @return バージョン：成功<br><br>
     * <b>サンプル：</b><br><hr>
     * <pre>{@code
     * import ZxingP5.*;
     * void setup(){
     *   size(400,400);
     *   println( CodeReader.getVersion() );
     * }
     * void draw(){
     *   background(255);
     * }
     * }
     * </pre>
     */
    public static String getVersion() {
        return VERSION;
    }
}
